#!/usr/bin/env perl

use strict;
use warnings;

use YAML qw/DumpFile LoadFile/;
use DBI;
use File::Copy;
use File::Copy::Recursive qw/dircopy/;
use Data::Dumper;

package ScaffoldSet;
use Moose;
use HTML::Template::Compiled;

has 'name'    => ( is => 'rw');
has 'columns' => ( isa => 'ArrayRef', is => 'rw', default => sub {[]} );

sub as_list {
    my $self = shift;
    my $o = join( ' ', grep { $_ ne 'ID' } @{$self->columns} );
    return $o;
}

sub to_perl {
    my $self = shift;
    my $htc = HTML::Template::Compiled->new(filename => 'code_templates/set.pl.tmpl');
    $htc->param( table => $self );
    return $htc->output;
}

1;

package ScaffoldRow;
use Moose;
use HTML::Template::Compiled;

has 'name'    => ( is => 'rw');
has 'columns' => ( isa => 'ArrayRef', is => 'rw', default => sub {[]} );

sub as_list {
    my $self = shift;
    my $o = join( ' ', grep { $_ ne 'ID' } @{$self->columns} );
    return $o;
}

sub as_self_list {
    my $self = shift;
    my $o = join(
        ", ", 
        map { '$self->' . $_ . '()' } 
        grep { $_ ne 'ID' } @{$self->columns} 
    );
    return $o;
}

sub as_param_list {
    my $self = shift;
    my $o = join(
        ", ", 
        map { "$_=(?)" } 
        grep { $_ ne 'ID' } @{$self->columns} 
    );
    return $o;
}

sub as_moose_properties {
    my $self = shift;
    my $o = join(
        "\n", 
        map { "has '$_' \t=>\t ( is => 'rw' );" } @{$self->columns} 
    );
    return $o;
}

sub to_perl {
    my $self = shift;
    my $htc = HTML::Template::Compiled->new(filename => 'code_templates/row.pl.tmpl');
    $htc->param( table => $self );
    return $htc->output;
}

1;

package ScaffoldCtrl;
use Moose;
use HTML::Template::Compiled;

has 'name'      => ( is => 'rw');
has 'dsn'       => ( is => 'rw');
has 'dbuser'    => ( is => 'rw');
has 'dbpasswd'  => ( is => 'rw');
has 'columns'   => ( isa => 'ArrayRef', is => 'rw', default => sub {[]} );

sub as_list {
    my $self = shift;
    my $o = join( ' ', grep { $_ ne 'ID' } @{$self->columns} );
    return $o;
}

sub to_perl {
    my $self = shift;
    my $htc = HTML::Template::Compiled->new(filename => 'code_templates/controller.pl.tmpl');
    $htc->param( table => $self );
    return $htc->output;
}

1;

package main;

# use GetOpt::Long;

my $config = LoadFile( "config/database.yaml" );
# warn Dumper $config;

my $table  = shift @ARGV || 'album'; # better use GetOpt::Long instead

my $dsn = "DBI:mysql:database=" 
        . $config->{schema} . ";host=" 
        . $config->{host} . ";port=3306";

my $dbh = DBI->connect($dsn, $config->{user}, $config->{passwd}) 
    or die "can't connect to database!";

my $cols = [ map { $_->[0] } @{$dbh->selectall_arrayref("DESCRIBE $table")} ];

mkdir './draft' or warn $!   unless -e './draft';
mkdir './draft/M' or warn $! unless -e './draft/M';
mkdir './draft/V' or warn $! unless -e './draft/V';
mkdir './draft/images' or warn $! unless -e './draft/images';

print "Creating MODEL ./draft/M/$table.pm ...\n";
open( my $r_fh, '>', "./draft/M/$table.pm" ) or warn $!;
    my $sc_row = ScaffoldRow->new( name => $table );
    $sc_row->columns($cols);
    print $r_fh $sc_row->to_perl;
close $r_fh;

print "Creating MODEL ./draft/M/set_$table.pm ...\n";
open( my $s_fh, '>', "./draft/M/set_$table.pm" ) or warn $!;
    my $sc_set = ScaffoldSet->new( name => $table );
    $sc_set->columns($cols);
    print $s_fh $sc_set->to_perl;
close $s_fh;

print "Creating CGI ./draft/$table.pl ...\n";
open( my $ctrl_fh, '>', "./draft/$table.pl" ) or warn $!;
    my $sc_ctrl = ScaffoldCtrl->new( name => $table, dsn => $dsn, 
        dbuser => $config->{'user'}, dbpasswd => $config->{'passwd'} );
    $sc_ctrl->columns($cols);
    print $ctrl_fh $sc_ctrl->to_perl;
close $ctrl_fh;

print "Creating VIEW ./draft/V/$table.tmpl ...\n";
copy("./view_templates/entity_base.tmpl","./draft/V/$table.tmpl") 
    or die "Copy failed: $!";

print "Copying VIEW to ./draft/V/nav.tmpl ...\n";    
copy("./view_templates/nav.tmpl","./draft/V/nav.tmpl") 
    or die "Copy failed: $!";

print "Copying CSS to ./draft/default.css ...\n";
copy("./css_templates/bright.css","./draft/default.css") 
    or die "Copy failed: $!";
    
print "Copying CSS to ./draft/dark.css ...\n";
copy("./css_templates/dark.css","./draft/dark.css") 
    or die "Copy failed: $!";
    
print "Copying images to ./draft/images/ ...\n";
dircopy("./images/", "./draft/images/") or die "Copy failed: $!";

print "Done!\n";
$dbh->disconnect;
