package CatcherInTheRye;

$CGI::POST_MAX = 1024 * 5000; # = 5MB

use strict;
use warnings;
use base("CGI::Application::Plugin::HTCompiled", "CGI::Application");

use CGI::Application::Plugin::Forward;
use CGI::Application::Plugin::Redirect;
use CGI::Application::Plugin::ValidateRM;
use CGI::Application::Plugin::ConfigAuto (qw/cfg/);
use CGI::Application::Plugin::Session;
use CGI::Application::Plugin::MessageStack;

use Data::Dumper qw/Dumper/;
use FileHandle;
use File::Find::Rule;
use File::Basename;
use MIME::Types;
use Data::FormValidator::Constraints::Upload qw(
	file_format
	file_max_bytes
	image_max_dimensions
	image_min_dimensions
);

=head1 NAME

CatcherInTheRye - a boring book.

=head1 SYNOPSIS

  use CatcherInTheRye;
  blah blah blah

=head1 DESCRIPTION

This is an example of an upload form. It's written in perl and uses
CGI::Application. It features CGI::Application::Dispatch, Data::FormValidator,
Data::FormValidator::Constraints::Upload, CGI::Application::MessageStack,
Find::File::Rule and MIME::Types.

This app has the folowing cmponents:

=over
=item upload form: user can specify a file to upload.
=item upload validation: vaiadtion of the user input including restriction of mime type, size and image dimensions.
=item uploaded file view: displays the contents of the folder where the files are stored. Files may be deleted or downloaded.
=item download button for files: Send a file trough the app. No need to point at the real folder.
=back

Blah blah blah.

=head2 EXPORT

None by default.


=head1 METHODS

=cut

=head2 cgiapp_init()

Open database connection, setup config files, etc.

=cut

sub cgiapp_init {
	my $self = shift;
	
	# -- Configure MessageStack.
	$self->capms_config(
		-automatic_clearing            => 1,
	);
	
	
    # -- Set some defaults for DFV unless they already exist.  
    $self->param('dfv_defaults') ||
        $self->param('dfv_defaults', {
                missing_optional_valid => 1,
                filters => 'trim',
                msgs => {
                    any_errors => 'some_errors',
                    prefix     => 'err_',
                    invalid    => 'Invalid',
                    missing    => 'Missing',
                    format => '<span class="dfv-errors">%s</span>',
                },
        });
	
	
	# -- Force reload every submission. usefull if in development environment.
	$self->header_props( -expires => 'now' );
	
} # /cgiapp_init




=head2 setup()

Defined runmodes, etc.

=cut

sub setup {
	my $self = shift;
	
	$self->start_mode('start');
	$self->mode_param('rm');
	$self->run_modes([qw/
		start
		form_validate
		show_uploaded_files
		file_delete
		file_download
	/]);
	
} # /setup




=head2 start( $errs? )

Display the upload form.

=cut

sub start {
	my $self = shift;
	my $errs = shift; # may be undef
	
	my $t = $self->load_tmpl('upload_form.tmpl');
	$t->param($errs) if $errs;
	return $t->output();
} # /start




=head2 form_validate()

Validate user input.
If valid, upload the file, redirect to success (form with happy message).
Unless valid, redirect to form, show errors.

=cut

sub form_validate {
	my $self = shift;
	my $upload_dir = $self->cfg('upload_dir') or die("Missing upload dir param.");
	
	my $form_profile = {
		required => [qw/new_file/],
		optional => [qw/submit rm/],
		constraint_methods => {
			new_file => [
				file_format(mime_types => [qw!image/jpeg image/gif image/png!]),
				file_max_bytes(1024000),
				image_max_dimensions(700,500),      
				image_min_dimensions(100,100),
			],
        }
	};
	
	my  $results = $self->check_rm('start', $form_profile) || return $self->check_rm_error_page();
	
	# -- Check fields for presence.
	unless( $results->valid('new_file') ) {
		return "missing file";
	}
	
	# -- Get the filename.
	my $insecure_filename = $results->valid('new_file');
	my $filename = $self->get_save_filename($insecure_filename);
	
	# -- Get the filehandle.
	my $upload_filehandle = $self->query()->upload("new_file");

	# -- Calculate filename.
	my $picture_uri = "$upload_dir/$filename";
	
	# -- Check if file exists.
	if( -e $picture_uri ) {
		# File exists.
		return "File exists.";
	}
	
	# -- Upload the file.
	my $fh = FileHandle->new();
	if( $fh->open($picture_uri, '>') ) {
		
		$fh->binmode();
		while( <$upload_filehandle> ) {
			$fh->print($_);
		}
		
		$fh->close();
		
	}else{
		die("Error opening [$picture_uri]: $!.");
	}
	
	# -- Set up a small confirmation message.
	$self->push_message(
		-scope          => 'start',
		-message        => 'Your file has been uploaded.',
		-classification => 'INFO',
    );
	
	return $self->redirect($self->query()->url());
} # /form_validate




=head2 show_uploaded_files()

Display a list of all uploaded files. Add some information like what kind of
file it is. Add a download button.

=cut

sub show_uploaded_files {
	my $self = shift;
	my $upload_dir = $self->cfg('upload_dir') or die("Missing upload dir param.");
	
	# Find all files in the upload dir.
	my @files_on_disk = File::Find::Rule->file()->in( $upload_dir );
	
	my $mt = MIME::Types->new();
	
	my @file_loop = ();
	foreach my $file ( @files_on_disk  ) {
		my $basename = File::Basename::basename($file);
		my $type = $mt->mimeTypeOf($file);
		push @file_loop, {
			filename => $basename,
			mime_type => $type,
			c => $self,
		};
	}
	
	my $t = $self->load_tmpl('file_list.tmpl');
	$t->param(files => \@file_loop) if @file_loop;
	return $t->output();
} # /show_uploaded_files




=head2 file_delete()

Delete a file from disk. Does the same file chacks as the upload form to
prevent hacky filenames.

=cut

sub file_delete {
	my $self		= shift;
	my $file		= $self->param('filename') or return("Missing filename.");
	my $upload_dir	= $self->cfg('upload_dir') or die("Missing upload dir param.");
	
	return "Will not delete [$file]. Go away." if $file ne $self->get_save_filename($file);
	
	my $file_uri = "$upload_dir/$file";
	
	if( !-e $file_uri or -d $file ) {
		return("File [$file_uri] does not exists.");
	}
	
	my $rv = unlink( $file_uri );
	
	if( $rv == 1 ) {
		
		# -- Set up a small confirmation message.
		$self->push_message(
			-scope          => 'show_uploaded_files',
			-message        => 'Your file has been deleted.',
			-classification => 'INFO',
		);
		
	}else{
		# Deletition didn't work.
		# -- You tell the user!
		$self->push_message(
			-scope          => 'show_uploaded_files',
			-message        => 'Your file has not been deleted. I dunno why.',
			-classification => 'ERROR',
		);
		
	}
	
	my $url = $self->query()->url() . '/' . __PACKAGE__ . '/show_uploaded_files';
	return $self->redirect($url);
} # /file_delete




=head2 file_download()

Provide a file for download. This simply sets the content type of the CGI header
to the mime type of the file and returns the file content without any other
data (e.g. the layout html in cgiapp_postrun).

=cut

sub file_download {
	my $self = shift;
	my $file		= $self->param('filename') or return("Missing filename.");
	my $upload_dir	= $self->cfg('upload_dir') or die("Missing upload dir param.");
	
	return "Will not provide this file: [$file]. Go away." if $file ne $self->get_save_filename($file);
	
	# -- Don't surround a data stream with a html layout.
	$self->param('skip_postrun' => 1);
	
	my $file_uri = "$upload_dir/$file";
	my $mto = MIME::Types->new(); # mto = mime type object
	my $mime_type = $mto->mimeTypeOf($file_uri);
	
	# -- Set the header to the mime type of the file.
	$self->header_props( -type => $mime_type );
	
	my $data = undef;
	my $fh = FileHandle->new();
	if( $fh->open($file_uri) ) {
		
		$fh->binmode();
		while( my $l = $fh->getline() ) {
			$data .= $l;
		}
		$fh->close();
		
	}else{
		die("ERROR: Cannot open $file_uri: $!");
	}
	
	binmode( STDOUT );
    return $data;
} # /file_download




=head2 get_save_filename( $filename )

Catch characters we don't want. Catch hacky filenames.

=cut

sub get_save_filename {
	my $self		= shift;
	my $filename	= shift or die("Missing filename.");
	my $upload_dir	= $self->cfg('upload_dir') or die("Missing upload dir param.");

	# -- Check filename for forbidden characters and other hacks.
	my ( $name, $path, $extension ) = fileparse ( $filename, '\..*' ); 
	$filename = $name . $extension;

	my $safe_filename_characters = "a-zA-Z0-9_.-";
	
	$filename =~ tr/ /_/;
	$filename =~ s/[^$safe_filename_characters]//g;
	
	return $filename;
} # /get_save_filename




=head2 cgiapp_postrun()

Output manipulation:

=over
=item add common header/footer?
=item Cleanup your HTML?
=item Rewrite URLs?
=back

=cut

sub cgiapp_postrun {
	my $self = shift;
	my $output = shift;

	return if $self->param('skip_postrun');
	
	my $layout = $self->load_tmpl('layout.tmpl');
	$layout->param('main_content' => $$output);
	$$output = $layout->output();

} # /cgiapp_postrun




=head1 SEE ALSO

Inspired by http://www.sitepoint.com/article/uploading-files-cgi-perl/2/

Shamelessly used the blue:blossom design done by http://www.jonasjohn.de/

=head1 AUTHOR

Alexander Becker E<lt>capfan@gmx.deE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2009 by Alexander Becker

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.

=cut

1;