#!/usr/bin/perl

use Time::HiRes;
use List::Util qw(shuffle);
use SDL::Events;
use SDL::Event;
use SDL::Image;
use SDL::Surface;
use SDLx::App;
use SDLx::LayerManager;
use SDLx::Layer;
use SDLx::FPS;

my $display         = SDLx::App->new(title => 'Games::Solitaire',
                                     width => 800, height => 600, depth => 32,
                                     flags => SDL_HWSURFACE | SDL_HWACCEL);
my $layers          = SDLx::LayerManager->new();
my $event           = SDL::Event->new();
my $loop            = 1;
my $last_click      = Time::HiRes::time;
my $fps             = SDLx::FPS->new(fps => 60);
my $left_mouse_down = 0;
my $handler         = {};

init_background();
init_cards();
$layers->blit($display);
$display->flip();
game();

sub event_loop {
    SDL::Events::pump_events();
    while(SDL::Events::poll_event($event)) {
        my $type = $event->type;

        if ($type == SDL_MOUSEBUTTONDOWN) {
            $left_mouse_down = 1 if $event->button_button == SDL_BUTTON_LEFT;
            my $time = Time::HiRes::time;
            if ($time - $last_click >= 0.3) {
                $handler->{on_click}->();
            }
            else {
                $handler->{on_dblclick}->();
            }
            $last_click = $time;
        }
        elsif ($type == SDL_MOUSEMOTION) {
            if ($left_mouse_down) {
                $handler->{on_drag}->();
            }
            else {
                $handler->{on_mousemove}->();
            }
        }
        elsif ($type == SDL_MOUSEBUTTONUP) {
            $left_mouse_down = 0 if $event->button_button == SDL_BUTTON_LEFT;
            $handler->{on_drop}->();
        }
        elsif ($type == SDL_KEYDOWN) {
            $handler->{on_keydown}->();
        }
        elsif ($type == SDL_QUIT) {
            $handler->{on_quit}->();
        }
    }
}

sub game {
    my @selected_cards = ();
    $handler = {
        on_quit      => sub {
            $loop = 0;
        },
        on_drag      => sub {
        },
        on_drop      => sub {
            if(scalar @selected_cards) {
                my $i = 0;
                for (@selected_cards) {
                    $selected_cards[$i] = $selected_cards[$i]->foreground;
                    $i++;
                }
                
                my $drop_target = $selected_cards[0]->behind->[0];
            
                if($drop_target->data->{id} =~ m/empty_stack/
                && $drop_target->pos->y >= 200
                && $selected_cards[0]->data->{id} % 13 == 12) {
                    $layers->detach_xy($drop_target->pos->x, $drop_target->pos->y);
                    @selected_cards = ();
                }
                elsif($drop_target->data->{id} =~ m/^\d+$/
                && $drop_target->data->{visible}
                && $drop_target->pos->y >= 200
                && $selected_cards[0]->data->{id} % 13 + 1 == $drop_target->data->{id} % 13
                && int($selected_cards[0]->data->{id} / 13) % 2 != int($drop_target->data->{id} / 13) % 2) {
                    $layers->detach_xy($drop_target->pos->x, $drop_target->pos->y + 20);
                }
                else {
                    $layers->detach_back;
                }
                
                @selected_cards = ();
            }
        },
        on_click     => sub {
            unless(scalar @selected_cards) {
                my $layer = $layers->by_position($event->button_x, $event->button_y);
                
                if(defined $layer
                && $layer->data->{id} =~ m/^\d+$/
                && $layer->data->{visible}) {
                    @selected_cards = ($layer, @{$layer->ahead});
                    $layers->attach(@selected_cards, $event->button_x, $event->button_y);
                }
            }
        },
        on_dblclick  => sub {
        },
        on_mousemove => sub {
        },
        on_keydown   => sub {
        },
    };

    while($loop) {
        event_loop();
        if(scalar @{$layers->blit($display)}) {
            $display->flip();
        }
        $fps->delay;
    }
}

sub can_drop {
    my $moved_layer = shift;
    my $drop_target = shift;
    if(($moved_layer->data->{id} + 1) % 13 + 1 == ($drop_target->data->{id} + 1) % 13) {
        return 1;
    }
    
    return 0;
}

sub init_background {

  # Hintergrundbild
  my $background  = SDL::Image::load('data/background.jpg');
  $layers->add(SDLx::Layer->new($background, {id => 'background'}));

  my $empty_stack = SDL::Image::load('data/empty_stack.png');

  # zwei Platzhalter fr Kartenstapel oben links
  $layers->add(SDLx::Layer->new($empty_stack,  20, 20, {id => 'rewind_deck'}));
  $layers->add(SDLx::Layer->new($empty_stack, 130, 20, {id => 'empty_deck'}));

  # sieben Stapel unten
  for(0..6) {
    $layers->add(SDLx::Layer->new($empty_stack, 20+110*$_, 200, {id => 'empty_stack'}));
  }

  # vier Stapel oben rechts
  for(0..3) {
    $empty_target = SDL::Image::load('data/empty_target_' . $_ . '.png');
    $layers->add(SDLx::Layer->new($empty_target, 350+110*$_, 20, {id=>"empty_target_$_"}));
  }
}

sub init_cards {
    my $stack_index    = 0;
    my $stack_position = 0;
    my @card_value     = shuffle(0..51);
    for(0..51)
    {
        my $image   = 'data/card_back.png';
        my $visible = 0;
        my $x       = 20;
        my $y       = 20;
        
        if($_ < 28)
        {
            if($stack_position > $stack_index)
            {
                $stack_index++;
                $stack_position = 0;
            }
            if($stack_position == $stack_index)
            {
                $image   = 'data/card_' . $card_value[$_] . '.png';
                $visible = 1;
            }
            $x =  20 + 110 * $stack_index;
            $y = 200 +  20 * $stack_position;
            $stack_position++;
        }
        
        $layers->add(SDLx::Layer->new(SDL::Image::load($image), $x, $y, {id => $card_value[$_], visible => $visible}));
    }
}

