package FooConfig;

use strict;
use warnings;

use Carp;
use DateTime;
use DateTime::Format::ICal;
use File::Basename;
use File::Spec;
use File::stat;
use List::Util qw/first/;
use Storable qw/lock_store lock_retrieve/;
use XML::LibXML;
use Data::Dumper;

sub new{
    my ($class,$file,$client,$cache) = @_;
    
    my $self = {};
    bless $self,$class;
    
    # check for needed params
    croak "path to file needed!"      unless $file;
    croak "path to cache dir needed!" unless $cache;
    croak "client name needed!"       unless $client;
    
    # save attributes
    $self->cache_dir( $cache );
    $self->client( $client );
    
    # get some general values
    my $time         = time;
    my $client_cache = $self->get_cache( $file, $time );
    
    # load appropriate file
    if ( $client_cache ) {
        $self->load_cache( $client_cache );
    }
    else {
        $self->load( $file );
    }

    return $self;
}

sub cache_dir {
    my ($self,$dir) = @_;
    
    $self->{_cache_dir} = $dir if @_ == 2;
    return $self->{_cache_dir};
}

sub client {
    my ($self,$client) = @_;
    
    $self->{_client_name} = $client if @_ == 2;
    return $self->{_client_name};
}

sub load{
    my ($self,$file) = @_;
    
    # check for needed parameter
    croak "no config file given" unless defined $file;
    
    my $parser = XML::LibXML->new;
    $parser->keep_blanks(0);
    my $tree   = $parser->parse_file( $file );

    my $root   = $tree->getDocumentElement;

    my $hashref = {};

    $self->walk_tree( $hashref, $root );
        
    $self->{_config} = $hashref;
    
    # try to determine, which config should be used (based on time)
    $self->{_config} = $self->select_by_time( $self->{_config} );
    
    $self->save_cache( $file );
    
    return $self->{_config};
}

sub save_cache {
    my ($self,$config) = @_;
    
    my $cache   = $self->cache_dir;
    my $client  = $self->client;
    my $hashref = $self->{_config};
    my $epoche  = $self->valid_to || 9999999999;

    return if !$epoche; 
    
    # save timestamp file for config file
    my $mtime       = stat( $config )->mtime;
    my $mtime_file  = File::Spec->catfile( $cache, $mtime . '_base.dat' );
    lock_store { mtime => $mtime }, $mtime_file;
    
    # save cache file
    my $cache_file = File::Spec->catfile( $cache, $client . '_' . $epoche . '.dat' );
    lock_store $hashref, $cache_file;
    
}

sub load_cache {
    my ($self,$cache_file) = @_;

    unless ( -r $cache_file ) {
        $self->{_config} = {};
        return;
    }
    
    my $hashref = lock_retrieve $cache_file;
    $self->{_config} = $hashref;
}

sub walk_tree {
    my ($self,$hashref,$node) = @_;
    
    my @children = $node->childNodes;
    
    CHILD:
    for my $child ( @children ) {
        my $name         = $child->nodeName;
        my @has_children = $child->childNodes;
        
        next CHILD unless @has_children;
        
        if ( $name eq 'kunde' ) {
            
            my $client = $self->client;
            $name = $child->getAttribute( 'name' );
            
            next CHILD if $name ne $client;
            
            for my $logonode ( @has_children ) {
                my $logo_config = $self->walk_tree( {}, $logonode );
                push @{ $hashref->{$name} }, $logo_config;
            }
            
            next CHILD;
        }
        
        $hashref->{$name} = {} unless $hashref->{$name};
        
        if ( my ($text) = grep{ $_->nodeName eq '#text' }@has_children ) {
            $hashref->{ $name } = $text->textContent;
            next CHILD;
        }
        else {
            $self->walk_tree( $hashref->{ $name }, $child );
        }
        
        if ( $name eq 'logo' ) {
            $hashref = $hashref->{logo};
        }
    }

    return $hashref;
}

sub get_cache {
    my ($self, $config, $time) = @_;

    my $dir    = $self->cache_dir;
    my $client = $self->client;

    # get all files in the directory
    my %files;
    if ( opendir my $dirhandle, $dir ) {
        %files = map{ $_ => $dir . '/' . $_ }
                 grep{ /\.dat \z/x and -f $dir . '/' . $_ }
                 readdir $dirhandle;
        closedir $dirhandle;
    }

    # delete config cache files that are outdated
    my @old_files = grep{ /(\d+)\.dat/ && $1 < $time }keys %files;


    unlink @files{@old_files};
    delete @files{@old_files};

    my $cache_file;

    # if a file with timestamp for config file exists
    if ( my $config_check = first{ /\A \d+ _base\.dat \z/x }keys %files ) {
        
        # Timestamp of the config file, the cache files are
        # based on.
        my ($check_timestamp) = $config_check =~ /\A(\d+)_/;

        # get "last modified" time of config file
        my $stat = stat( $config )->mtime;

        # if cache files do not contain current info
        # delete them
        if ( $check_timestamp != $stat ) {
            unlink values %files;
            %files = ();
        }
        else {
            
            # does a cache file for the requested client exist?
            my $possible_cache    = first{ /\A $client _ \d+ \.dat \z/x }keys %files;
            $possible_cache     ||= '';
            my ($check_timestamp) = $possible_cache =~ / (\d+) \.dat /x;
            
            # cache file is still valid
            if ( $possible_cache and $check_timestamp > $time ) {
                $cache_file   = $possible_cache;
            }
        }
    }
    
    return $files{$cache_file} if $cache_file;
    return;
}

sub select_by_time {
    my ($self,$config) = @_;
    
    my @to_delete;
    my $requested_client = $self->client;

    if ( exists $config->{kunden} ) {
        
        client:
        for my $client ( keys %{ $config->{kunden} } ) {
            
            next client if !$client;
            next client if $requested_client and $client ne $requested_client;
            
            if ( ref( $config->{kunden}->{$client} ) eq 'ARRAY' ) {
                my $default;
                my $local_config;
                
                PART:
                for my $part ( @{ $config->{kunden}->{$client} } ) {

                    next PART if !$part or ref( $part ) ne 'HASH';
                    
                    # check if all needed info is given
                    if ( !exists $part->{vevent} ) {
                        $default = $part;
                        next PART;
                    }
                    elsif ( exists $part->{vevent} && ref( $part->{vevent} ) eq 'HASH' &&
                            !( $part->{vevent}->{dtstart} && $part->{vevent}->{dtend}
                                    && $part->{vevent}->{rrule} ) ) {
                        next PART;
                    }

                    my $start      = $part->{vevent}->{dtstart};
                    my $end        = $part->{vevent}->{dtend};
                    my $recurrence = $part->{vevent}->{rrule};
                    
                    my ($date,$end_obj,$rec_obj);
                    
                    eval {
                        $date       = DateTime::Format::ICal->parse_datetime( $start );
                        $end_obj    = DateTime::Format::ICal->parse_datetime( $end );
                        $rec_obj    = DateTime::Format::ICal->parse_recurrence(
                            recurrence => $recurrence,
                            dtstart    => $date,
                        );
                        1;
                    };
                                        
                    next PART if !( $date && $end_obj && $rec_obj );

                    my $today   = DateTime->now;
                    my $dur_obj = $end_obj - $date;
                    
                    if ( $self->contains( $rec_obj, $dur_obj, $today ) ) {
                        $local_config = $part;
                        last PART;
                    }
                }
                
                $config->{kunden}->{$client} = $local_config || $default;
                
                if ( !$config->{kunden}->{$client} ) {
                    push @to_delete, $client;
                }
            }
        }
    }
    
    delete @{$config->{kunden}}{@to_delete};

    $config;
}

sub contains {
    my ($self,$rec,$dur,$day) = @_;
    
    my $return = 0;
    
    my $iter = $rec->iterator;
    while ( my $dt = $iter->next ) {
        last if $dt > $day;
        
        my $end = $dt + $dur;
        
        next if $end < $day;
        if ( $end >= $day and $dt <= $day ) {
            $return = 1;
            $self->valid_to( $end->epoch );
        }
    }
    
    return $return;
}

sub valid_to {
    my ($self,$epoch) = @_;
    
    $self->{__epoch} = $epoch if @_ == 2;
    $self->{__epoch};
}

sub get {
    my ($self,$key) = @_;

    my $return;

    if( defined $key ){
        my $config = $self->{_config};

        my @keys = split /(?<!\\)\./, $key;
        for my $subkey ( @keys ){
            $subkey =~ s/\\\././g;
            return if not exists $config->{$subkey};
            $config = $config->{$subkey};
        }

        $return = $config;
    }

    $return;
}

1;
